/**
 * URM – User List
 *
 * Displays a paginated list of users with their assigned membership roles,
 * including start and expiration dates. Allows administrators to search,
 * revoke roles, export user data to CSV, and clean expired role records.
 *
 * Author: Black & Copilot
 */

// 👥 URM - User List

function render_ur_user_list_tab($frontend = false) {
	if (!function_exists('get_users')) {
    echo '<p style="color:red;">⛔ WordPress is not ready — get_users() is not available.</p>';
    return;
    }

    // ✅ Show messages outside of the wrapper
    if (isset($_GET['revoked'])) {
        $revoked_user_id = intval($_GET['revoked']);
        $msg_class = $frontend
            ? 'style="margin: 0 20px 20px 20px; background: #dff0d8; border: 1px solid #d0e9c6;
                padding: 12px; border-radius: 4px; color: #3c763d;"'
            : 'class="notice notice-success is-dismissible"';
        echo '<div ' . $msg_class . '><p>✅ Role for user ID ' . $revoked_user_id . ' has been revoked.</p></div>';
    }

    if (isset($_GET['cleaned'])) {
        $n = intval($_GET['cleaned']);
        $msg_class = $frontend
            ? 'style="margin: 0 20px 20px 20px; background: #dff0d8; border: 1px solid #d0e9c6;
                padding: 12px; border-radius: 4px; color: #3c763d;"'
            : 'class="notice notice-success is-dismissible"';
        echo '<div ' . $msg_class . '><p>🧹 User removed: <strong>' . $n . '</strong></p></div>';
    }

    // 👇 Enter the main block of the HTML.
    echo '<div style="margin-top: 0px;' . ($frontend ? '' : ' background: #fff;') . ' border: 1px solid #ccc; border-radius: 0px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); overflow: hidden;">';
	
    // 👑 Block title
    echo '<div style="' . ($frontend ? '' : 'background: #f7f9fa;') . ' padding: 10px 16px; border-bottom: 1px solid #ccc; font-weight: bold; font-size: 16px;">👥 User record with role changes</div>';

    // 🔍 Search
    $search = trim($_GET['search_user'] ?? '');
    echo '<div style="padding: 20px 20px 10px 20px;">';
    echo '<form method="get" action="' . ($frontend ? esc_url(get_permalink()) : '') . '" style="display: flex; gap: 10px; align-items: center;">';
    if (!$frontend) {
    echo '<input type="hidden" name="page" value="ur-membership">';
    echo '<input type="hidden" name="tab" value="users">';
    }
    echo '<input type="text" name="search_user" value="' . esc_attr($search) . '" placeholder="Search users by ID, username, or email..." style="flex: 1; padding: 8px; border: 1px solid #ccc; border-radius: 4px;">';
    echo '<button type="submit" style="padding: 8px 16px; background: #0073aa; color: #fff; border: none; border-radius: 4px; cursor: pointer;">🔎 Search users</button>';
    echo '</form>';
    echo '</div>';

	// 📊 Style and preparation
	echo '<div style="padding: 10px 20px 20px 20px;">';
    echo '<style>
        .urm-user-table th,
        .urm-user-table td {
            padding: 6px 12px;
            border: 1px solid #ccc;
            font-size: 14px;
            vertical-align: middle;
            word-break: break-word;
        }

        /* 🌓 Stripe colors according to context */
        .urm-user-table tr:nth-child(even) {
            background-color: ' . ($frontend ? 'inherit' : '#f9f9f9') . ';
        }
        .urm-user-table tr:nth-child(odd) {
            background-color: ' . ($frontend ? 'inherit' : '#ffffff') . ';
        }

        /* ✨ Hover effect front-end friendly */
        .urm-user-table tr:hover {
            background-color: ' . ($frontend ? 'rgba(255,255,255,0.05)' : '#eee') . ';
        }
    </style>';

    // 🧾 Table
    echo '<table class="urm-user-table" style="width: 100%; border-collapse: collapse; margin-top: 10px;">';
    echo '<thead><tr>
        <th>🆔 ID</th>
        <th>🖼️ Avatar</th>
        <th>👤 Username</th>
        <th>📧 Email</th>
        <th>⏱️ Role start date</th>
        <th>⏳ Role expiration date</th>
        <th>🎭 Active role</th>
        <th>⚙️ Revoke role</th>
    </tr></thead>';
    echo '<tbody>';

    // 📋 Pages and retrieval
    $per_page = isset($_GET['per_page']) ? max(1, intval($_GET['per_page'])) : 8;
    $paged = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
    $offset = ($paged - 1) * $per_page;

    $all_users = get_users(['meta_key' => 'ur_role_history', 'meta_compare' => 'EXISTS', 'number' => -1]);
    $total_users = count($all_users);
    $rows = [];

    foreach ($all_users as $user) {
        if ($search) {
            $match = false;
            if (stripos((string)$user->ID, $search) !== false) $match = true;
            elseif (stripos($user->user_login, $search) !== false) $match = true;
            elseif (stripos($user->user_email, $search) !== false) $match = true;
            if (!$match) continue;
        }

        $history = get_user_meta($user->ID, 'ur_role_history', true);
        if (!is_array($history)) continue;

        foreach ($history as $entry) {
            if (!isset($entry['start'], $entry['end'], $entry['role'])) continue;
            $rows[] = [
                'user'    => $user,
                'entry'   => $entry,
                'latest'  => $entry['start'] ?? 0
            ];
        }
    }

    // ⏱️ Sort by date
    usort($rows, fn($a, $b) => $b['latest'] <=> $a['latest']);
    $total_pages = ceil(count($rows) / $per_page);
    $paginated_rows = array_slice($rows, $offset, $per_page);
    $rows_rendered = 0;

    foreach ($paginated_rows as $row) {
        $user = $row['user'];
        $entry = $row['entry'];

        $start = date('d.m.Y. H:i:s', $entry['start']);
        $end   = date('d.m.Y. H:i:s', $entry['end']);
        $role = format_role_name($entry['role']);
        $duration = $entry['duration'];

        echo '<tr>';
        echo '<td>' . $user->ID . '</td>';
        echo '<td><center>' . get_avatar($user->ID, 40) . '</center></td>';
        echo '<td><a href="' . esc_url(get_author_posts_url($user->ID)) . '" target="_blank">' . esc_html($user->user_login) . '</a></td>';
        echo '<td><a href="mailto:' . esc_attr($user->user_email) . '">' . esc_html($user->user_email) . '</a></td>';
        echo '<td>' . $start . '</td>';
        echo '<td>' . $end . '</td>';
        echo '<td>' . esc_html(format_role_name($user->roles[0] ?? 'N/A')) . '</td>';

        // 🔘 Manual revocation
        echo '<td>';
        if (function_exists('user_has_active_membership_role') && user_has_active_membership_role($user->ID)) {
            echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" onsubmit="return confirm(\'⚠️ Are you sure you want to revoke this user role?\');">';
            echo '<input type="hidden" name="action" value="revoke_user_role">';
            echo '<input type="hidden" name="user_id" value="' . intval($user->ID) . '">';
            echo '<input type="hidden" name="redirect_to" value="' . esc_url($_SERVER['REQUEST_URI']) . '">';
            echo '<center><button type="submit" style="background: #dc3545; color: white; border: none; padding: 6px 12px; border-radius: 4px; cursor: pointer;">⛔ Revoke role</button></center>';
            echo '</form>';

        } else {
            echo '<center><span style="color: gray;">– – – – –</span></center>';
        }
        echo '</td>';

        echo '</tr>';
        $rows_rendered++;
    }

    if ($rows_rendered === 0) {
        echo '<tr><td colspan="8" style="text-align: center; color: gray;">There are no results to display.</td></tr>';
    }

    echo '</tbody></table>';

    // 🔢 Navigation
    if ($total_pages > 1) {
        echo '<div style="margin-top: 15px;">';
        for ($i = 1; $i <= $total_pages; $i++) {
    $query = [
        'paged' => $i,
        'per_page' => $per_page,
        'search_user' => $search
    ];
    $url = $frontend ? add_query_arg($query, get_permalink()) : add_query_arg($query);
    $active = ($i === $paged) ? 'font-weight: bold;' : '';
    echo '<a href="' . esc_url($url) . '" style="margin-right:10px; ' . $active . '">Page ' . $i . '</a>';
}
        echo '</div>';
    }

    // 🎛️ Number of entries per page
    echo '<form method="get" action="' . ($frontend ? esc_url(get_permalink()) : esc_url(admin_url('admin.php?page=ur-membership&tab=users'))) . '" style="margin-top: 20px; display: inline-block;">';
if (!$frontend) {
    echo '<input type="hidden" name="page" value="ur-membership">';
    echo '<input type="hidden" name="tab" value="users">';
}
    echo '<input type="hidden" name="search_user" value="' . esc_attr($search) . '">';
    echo 'Show <input type="number" name="per_page" value="' . esc_attr($per_page) . '" min="5" max="500" style="width:60px;"> entry by page ';
    echo '<button type="submit" style="margin-left:10px;  background: #198754; color: white; border: none; padding: 8px 20px; border-radius: 4px; cursor: pointer;">🔄 Apply</button>';
    echo '</form>';

    // ⬇️ CSV export
    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="margin-top: 20px; display: inline-block; margin-left: 20px;">';
    echo '<input type="hidden" name="action" value="export_ur_user_list">';
    echo '<button type="submit" style="background: #0d6efd; color: white; border: none; padding: 8px 20px; border-radius: 4px; cursor: pointer;">⬇️ Export CSV</button>';
    echo '</form>';
	
	// 🗑️ Clean the list of users whose roles have expired
	echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="margin-top: 20px; display: inline-block; margin-left: 20px;" onsubmit="return confirm(\'⚠️ Are you sure you want to delete users with expired roles?\');">';
    echo '<input type="hidden" name="action" value="ur_user_list_cleanup">';
    echo '<button type="submit" style="background: #dc3545; color: white; border: none; padding: 8px 20px; border-radius: 4px; cursor: pointer;">🧹 Clean expired roles</button>';
    echo '</form>';


    echo '</div>'; // close .padding
    echo '</div>'; // close .frame
}

add_shortcode('ur_user_list', function () {
    if (!current_user_can('manage_options')) {
        return '<center><div style="color:red; font-weight:bold;">⛔ This page is only available to administrators.</div></center>';
    }
    ob_start();
	
    render_ur_user_list_tab(true);
    return ob_get_clean();
});

add_action('admin_post_export_ur_user_list', function () {
    if (!current_user_can('manage_options')) wp_die('⛔ You do not have sufficient permissions.');

    $users = get_users(['meta_key' => 'ur_role_history', 'meta_compare' => 'EXISTS', 'number' => -1]);
    if (empty($users)) wp_die('📭 There are no users for export.');

    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="ur_user_list.csv"');

    $output = fopen('php://output', 'w');
    fputcsv($output, ['ID', 'Username', 'Email', 'Start', 'Expiration', 'Role', 'Duration']);

    foreach ($users as $user) {
        $history = get_user_meta($user->ID, 'ur_role_history', true);
        if (!is_array($history)) continue;
        foreach ($history as $entry) {
            fputcsv($output, [
                $user->ID,
                $user->user_login,
                $user->user_email,
                date('d.m.Y.', $entry['start']),
                date('d.m.Y.', $entry['end']),
                format_role_name($entry['role']),
                $entry['duration']
            ]);
        }
    }

    fclose($output);
    exit;
});

add_action('admin_post_revoke_user_role', function () {
    if (!current_user_can('manage_options')) wp_die('⛔ You do not have sufficient permissions.');

    $user_id = intval($_POST['user_id'] ?? 0);
    if (!$user_id) wp_die('⛔ Invalid user.');

    // 🧠 Debug check
    add_to_ur_process_log("🧠 Revert correction — user_id: {$user_id}", 'info');

    // 🔪 Attempt to recall the scheduled cron job
    $event = wp_get_scheduled_event('reset_user_role', [$user_id]);
    if ($event) {
        wp_unschedule_event($event->timestamp, 'reset_user_role', [$user_id]);
        add_to_ur_process_log("🔧 Cron role reset removed for user ID $user_id.", 'info');
    } else {
        add_to_ur_process_log("ℹ️ Cron for user ID $user_id not found or has already been executed.", 'info');
    }

    // ♻️ Reset role
    $user = new WP_User($user_id);
    $user->set_role('subscriber');

    // 📝 Log manual recall
    add_to_ur_process_log("⛔ Manual revoked of privileged role for user ID $user_id.", 'warning');

    $redirect_url = $_POST['redirect_to'] ?? ($_SERVER['REQUEST_URI'] ?? admin_url('admin.php?page=ur-membership&tab=users'));
    $redirect_url = add_query_arg('revoked', $user_id, $redirect_url);
    wp_redirect($redirect_url);
    exit;

});

add_action('admin_post_ur_user_list_cleanup', function () {
    if (!current_user_can('manage_options')) wp_die('⛔ You do not have sufficient permissions.');

    $users = get_users([
        'meta_key' => 'ur_role_history',
        'meta_compare' => 'EXISTS',
        'number' => -1
    ]);

    $cleared = 0;

    foreach ($users as $user) {
        $history = get_user_meta($user->ID, 'ur_role_history', true);
        if (!is_array($history) || empty($history)) continue;

        // If the user no longer has an active privileged role - they may be cleaned.
        if (function_exists('user_has_active_membership_role') && user_has_active_membership_role($user->ID)) {
            continue;
        }

        // Additional check: we do not touch users who still have multiple roles except 'subscriber'
        $roles = $user->roles;
        if (count($roles) === 1 && $roles[0] === 'subscriber') {
            delete_user_meta($user->ID, 'ur_role_history');
            add_to_ur_process_log("🧹 Cleared record for user ID {$user->ID}", 'info');
            $cleared++;
        }
    }

    wp_redirect(add_query_arg(['cleaned' => $cleared], wp_get_referer() ?: admin_url()));
    exit;
});
